#!/usr/bin/python3
#-*- coding: utf8 -*-

# @author : Sébastien LOZANO

"""Contient toutes les méthodes demandées et quelques unes écrites pour l’occasion.
  
  `Bibliothèques utiles`

      * csv
      * operator, uniquement sa méthode itemgetter
  
  `Fonctions demandées`

      * Extraire la sous-liste des établissements d’une commune
      donnée (colonne K). K est la 11eme lettre de l’alphabet
      c’est à l’index 10 d’un tableau.
      * Renvoyer la liste du nombre d’établissements privés et publics
      (colonne E) de chaque département (colonne W), sous la forme
      d’une liste de tuples:
      (département, nb d’établissements publics, nb d’établissements privés)
      * Renvoyer la sous-liste des établissements d’un département
      donné (colonne W).
      * Renvoyer la sous-liste des établissements ouverts avant une date donnée
      (colonne AI) sous la forme « JJ/MM/AAAA ».

    `Repérer les colonnes du fichier csv` 

      * 0  numero_uai
      * 1  appelation_officielle
      * 4  secteur_public_prive_libe
      * ...
      * 10 libelle_commune 
      * ...

"""

############################################################################################################
# Importation des bibliotheques
############################################################################################################
import csv # Pour la gestion des fichiers csv
from operator import itemgetter # Pour la gestion du tri des listes de tuples


############################################################################################################
# Extraction des données et découpage en sous-listes
# On extrait le tableau global dans une liste de tuples
# On découpe ce gros tableaux de 60 096 lignes en 20 sous-tableaux pour les tests
############################################################################################################

def cut_list(mylist: list, nb_sub_list: int) -> list:
  """Divise une liste en **nb_sub_list** sous-listes.        

    `Paramètres`

        * mylist : Une liste.
        * nb_sub_list : Le nombre de sous-listes souhaité.

    `Sorties`

        * out_list : Une liste de listes.
    
    `Notes`
    
        * Cette fonction est utile pour diviser les grandes listes de listes.

        * Si 'nb_sub_list' est supérieur au nombre d’éléments de la list,
        les dernières sous listes sont vides et chaque sous liste ne contient
        qu’un seul élément.

        * Si le nombre d’éléments de la list n’est pas un multiple
        de 'nb_sub_list' la dernière liste n’a pas le même nombre
        d’éléments que les autres.
        Elle contient le reste de la division euclidienne du nombre d’éléments
        de la liste à diviser par (nombre de sous-liste -1).

    `Exemples`
    
    >>> my_list_to_cut = [1,2,3,4,5,6,7,8,9,10,11]
    >>> cut_list(my_list_to_cut,4)
    [[1,2,3],[4,5,6],[7,8,9],[10,11]]
    >>> cut_list(my_list_to_cut,5)
    [[1, 2], [3, 4], [5, 6], [7, 8], [9, 10,11]]
    >>> cut_list(my_list_to_cut,6)
    [[1, 2], [3, 4], [5, 6], [7, 8], [9, 10], [11]]
    >>> cut_list(my_list_to_cut,7)
    [[1], [2], [3], [4], [5], [6], [7, 8, 9, 10, 11]]
    >>> cut_list(my_list_to_cut,13)
    [[1], [2], [3], [4], [5], [6], [7], [8], [9], [10], [11], [], []]    
  """
  pass

  nb_elt_sub_list = len(mylist)//(nb_sub_list-1)
  out_list = []

  if (nb_sub_list>len(mylist)):
    for i in range(nb_sub_list):
      out_list.append(mylist[i:i+1])
  else:
    for i in range(nb_sub_list-1):
      out_list.append(mylist[i*nb_elt_sub_list:(i+1)*nb_elt_sub_list])
    out_list.append(mylist[(nb_sub_list-1)*nb_elt_sub_list:])
  
  return out_list

#############  TESTS #########################
#print(cut_list([1,2,3,4,5,6,7,8,9],5))
##############################################

def extract_donnees_csv(path:str) -> list:
  """Une fonction pour extraire les données d’un fichier csv sous forme d'une liste de tuples.

    `Paramètres`    

        * path : Chemin vers le fichier csv.

    `Sorties`    

        * local_list_of_tuples_tab : Une liste de tuples.
          
    `Exemples`    

    >>> path_to_tab_etabs = './data/etabs_1er_2e_deg_prod.csv'
    >>> global_list_of_tuples_tab = extract_donnees_csv(path_to_tab_etabs) 
    >>> print(global_list_of_tuples_tab[1:3])
    >>> print(len(global_list_of_tuples_tab))
    [('0490005P', 'Lycée professionnel Simone Veil', 'LYCEE PROFESSIONNEL', 'SIMONE VEIL', 'Public', '77 rue DE SALPINTE', '', '61004', '49015', 'ANGERS CEDEX 01', 'Angers', '432487.8', '6711374.0', 'EPSG:2154', '47.44857212735416', '-0.5512003228265708', 'Correcte', 'Numéro de rue', '320', 'LYCEE PROFESSIONNEL', '1', 'OUVERT', '49', '52', '17', '49007', 'Maine-et-Loire', 'Pays de la Loire', 'Nantes', '47.4485721274,-0.551200322827', '99', 'SANS OBJET', '6', "MINISTERE DE L'EDUCATION NATIONALE", '1965-05-01'), ('0490048L', 'Collège Jean Rostand', 'COLLEGE', 'JEAN ROSTAND', 'Public', '142 avenue de la République', '', '', '49800', 'TRELAZE', 'Trélazé', '436341.5', '6711287.7', 'EPSG:2154', '47.44934443235111', '-0.5000595269804583', 'Parfaite', 'Numéro de rue', '340', 'COLLEGE', '1', 'OUVERT', '49', '52', '17', '49353', 'Maine-et-Loire', 'Pays de la Loire', 'Nantes', '47.4493444324,-0.50005952698', '99', 'SANS OBJET', '6', "MINISTERE DE L'EDUCATION NATIONALE", '1965-05-01')]
    66096
  """
  pass

  local_list_of_tuples_tab = []
  # On ouvre le fichier *.csv en lecture
  with open(path, newline='') as read_tab:
      # On passe l'objet à la méthode reader() pour récupérer l'objet reader 
      csv_reader = csv.reader(read_tab, delimiter=';')
      # On recupère toutes les lignes du fichier *.csv à partir de l'objet csv_reader sous forme d'une liste de tuples
      for row in csv_reader:      
          local_list_of_tuples_tab.append(tuple(row))
  return local_list_of_tuples_tab

# On extrait les données du fichier csv
path_to_tab_etabs = './data/etabs_1er_2e_deg_prod.csv'
global_list_of_tuples_tab = extract_donnees_csv(path_to_tab_etabs)

# Pour les tests, on découpe en 20 morceaux
# Pour les tests on n'utilisera que la première sous liste
# Cette sous-liste n'est utilisée que dans les tests de ce sous module PY_functions.py
# Ces tests sont commentés, il suffit de décommenter en cas de besoin
list_of_tuples_tab = cut_list(global_list_of_tuples_tab,20)[0]

#############  TESTS #########################
# Tableau global 60 096 lignes
#print(global_list_of_tuples_tab[1:3])
#print(len(global_list_of_tuples_tab))

# Tableau partiel, le 1er des 20
#print(list_of_tuples_tab[1:3])
#print(len(list_of_tuples_tab))
##############################################

############################################################################################################
# Avant les fonctions demandées, 
# Des fonctions utiles pour les fonctions demandées !
############################################################################################################

def extract_col(tab: list,col: int)->tuple:
  """Fonction pour extraire les éléments d’une colonne donnée.

    `Paramètres`

        * tab : Une liste.
        * col : L’index de la colonne dans laquelle se trouvent
        les données à extraire.

    `Sorties`

        * sortie : Les données contenues dans la colonnes.
    
    `Notes`

        * 0  numero_uai
        * 1  appelation_officielle
        * 4  secteur_public_prive_libe
        * ...
        * 10 libelle_commune 
        * ...
    
    `Exemples` : 
    
    Extraire l’entête et les 4 premiers numéros uai des établissements

    >>> path_to_tab_etabs = './data/etabs_1er_2e_deg_prod.csv'
    >>> global_list_of_tuples_tab = extract_donnees_csv(path_to_tab_etabs)
    >>> print(extract_col(global_list_of_tuples_tab[0:5],0))
    ('numero_uai', '0490005P', '0490048L', '0491025Y', '0491966W')

    Extraire l’entête et les 4 premiers libellés des communes des établissements

    >>> path_to_tab_etabs = './data/etabs_1er_2e_deg_prod.csv'
    >>> global_list_of_tuples_tab = extract_donnees_csv(path_to_tab_etabs)
    >>> print(extract_col(global_list_of_tuples_tab[0:5],10))
    ('libelle_commune', 'Angers', 'Trélazé', 'Cholet', 'Angers')
  """
  pass

  # un tuple pour récupérer les élement en sortie
  sortie = ()
  # On parcourt le tableau et on récupère l'index col de chaque élément
  for elt in tab:
    #print(elt)
    sortie = sortie + (elt[col],)
  return sortie

#############  TESTS ####################################
# Des tests avec le tableau global
# On extrait les numéros_uai
#print(extract_col(global_list_of_tuples_tab[0:5],0))
# On extrait les libellés des communes
#print(extract_col(global_list_of_tuples_tab[0:5],10))
#########################################################

def extract_col_critere(tab: list,col: int,critere: str,col_crit: int)->list:
  """Fonction pour extraire les éléments d’une colonne donnée via un critère donné.

    `Paramètres`

        * tab : Une liste.
        * col : L’index de la colonne à extraire.
        * critere : Une chaine de caractères avec le critère de séléction.
        Les valeurs possibles pour le critère doivent être récupérées 
        au préalable dans des tuples par exemple.
        * col_crit : L’index de la colonne contenant les éléments à extraire.

    `Sorties`

        * sortie : Une liste avec les éléments souhaités.
    
    `Notes`

        * 0  numero_uai
        * 1  appelation_officielle
        * 4  secteur_public_prive_libe
        * ...
        * 10 libelle_commune 
        * ...
        * 22 code_departement
    
    `Exemples`

    Exemple 1, on extrait les noms des établissements situés à Angers, on n'affiche que les 4 premiers sur les 125 au total :
    >>> path_to_tab_etabs = './data/etabs_1er_2e_deg_prod.csv'
    >>> global_list_of_tuples_tab = extract_donnees_csv(path_to_tab_etabs)     
    >>> print(extract_col_critere(global_list_of_tuples_tab,10,'Angers',1)[0:5])
    >>> print(len(extract_col_critere(global_list_of_tuples_tab,10,'Angers',1)))
    ['Lycée professionnel Simone Veil', 'Lycée général et technologique Henri Bergson', 'Ecole maternelle Alfred de Musset', 'INSTITUTION MARIA MONTESSORI', 'Ecole primaire Condorcet']
    125

    Exemple 2, on extrait le secteur public/privé des établissements situés à Angers, on n'affiche que les 4 premiers sur les 125 au total :
    >>> path_to_tab_etabs = './data/etabs_1er_2e_deg_prod.csv'
    >>> global_list_of_tuples_tab = extract_donnees_csv(path_to_tab_etabs)     
    >>> print(extract_col_critere(global_list_of_tuples_tab,10,'Angers',1)[0:5])
    >>> print(len(extract_col_critere(global_list_of_tuples_tab,10,'Angers',1)))
    ['Public', 'Public', 'Public', 'Privé', 'Public']
    125
  """
  pass

  # une liste pour récupérer les élement en sortie
  sortie = []  
  #compteur pour recuperer les index
  i=0
  # On parcourt le tableau et on récupère les index des éléments qui correspondent au critère
  tab_col = extract_col(tab,col)
  for elt in tab_col:   
    if (elt == critere):      
      sortie.append(tuple(tab[i])[col_crit])
    i+=1     
  return sortie

################################  TESTS ####################################
# Des tests avec le fichier global_list_of_tuples_tab

# exemple 1, on extrait les noms des établissements situés à Angers,
# on n'affiche que les 4 premiers et le nombre total
#print(extract_col_critere(global_list_of_tuples_tab,10,'Angers',1)[0:5])
#print(len(extract_col_critere(global_list_of_tuples_tab,10,'Angers',1)))

# exemple 2, on extrait le secteur public/privé des établissements 
# situés à Angers, on n'affiche que les 4 premiers et le nombre total
#print(extract_col_critere(global_list_of_tuples_tab,10,'Angers',4)[0:5])
#print(len(extract_col_critere(global_list_of_tuples_tab,10,'Angers',4)))
############################################################################

############################################################################################################
# Extraire la sous-liste des établissements d’une commune donnée (colonne K) 
# K est la 11eme lettre de l'alphabet c'est à l'index 10 d'un tableau
############################################################################################################
def extract_etabs_commune(tab: list,commune: str)->list:
  """Fonction pour extraire la sous liste des établissements d’une commune donnée.

    `Paramètres`

        * tab : Une liste de tuples issue de l'extraction du fichier csv.
        * commune : Une chaîne de caractères avec le nom de la commune.

    `Sorties`

        * extract_col_critere(tab,10,commune,1) : Une liste avec les noms des établissements de la commune.
    
    `Notes`
      
      On gère la casse car dans le fichier csv le nom des communes est formatté.
      La première lettre est en majuscule, les autres sont en minuscules.
      On a besoin de ce format pour passer l'argument à la fonction extract_col()

        * 0  numero_uai
        * 1  appelation_officielle
        * 4  secteur_public_prive_libe
        * ...
        * 10 libelle_commune 
        * ...
        * 22 code_departement
    
    `Exemples`

    On teste sur une partie du tableau global pour gagner du temps car le fichier csv global compte 
    60 096 lignes. On coupe le fichier global en 20 sous-tableaux.
    On ne traite donc que les 3 478 premières lignes pour l'exemple.
    Quelle que soit la casse entrée pour Angers, le tableau de sortie est le même !

    >>> path_to_tab_etabs = './data/etabs_1er_2e_deg_prod.csv'
    >>> global_list_of_tuples_tab = extract_donnees_csv(path_to_tab_etabs)  
    >>> list_of_tuples_tab = cut_list(global_list_of_tuples_tab,20)[0]
    >>> print(len(list_of_tuples_tab))
    >>> print(extract_etabs_commune(list_of_tuples_tab,'aNGERS'))
    >>> print(extract_etabs_commune(list_of_tuples_tab,'angerS'))
    >>> print(extract_etabs_commune(list_of_tuples_tab,'ANGERS'))
    >>> print(extract_etabs_commune(list_of_tuples_tab,'angers'))
    3478
    ['Lycée professionnel Simone Veil', 'Lycée général et technologique Henri Bergson', 'Ecole maternelle Alfred de Musset', 'INSTITUTION MARIA MONTESSORI', 'Ecole primaire Condorcet', 'Ecole élémentaire privée Immaculée-Conception', 'Ecole primaire privée Saint Laud', 'Ecole primaire privée Sainte Bernadette']
    ['Lycée professionnel Simone Veil', 'Lycée général et technologique Henri Bergson', 'Ecole maternelle Alfred de Musset', 'INSTITUTION MARIA MONTESSORI', 'Ecole primaire Condorcet', 'Ecole élémentaire privée Immaculée-Conception', 'Ecole primaire privée Saint Laud', 'Ecole primaire privée Sainte Bernadette']
    ['Lycée professionnel Simone Veil', 'Lycée général et technologique Henri Bergson', 'Ecole maternelle Alfred de Musset', 'INSTITUTION MARIA MONTESSORI', 'Ecole primaire Condorcet', 'Ecole élémentaire privée Immaculée-Conception', 'Ecole primaire privée Saint Laud', 'Ecole primaire privée Sainte Bernadette']
    ['Lycée professionnel Simone Veil', 'Lycée général et technologique Henri Bergson', 'Ecole maternelle Alfred de Musset', 'INSTITUTION MARIA MONTESSORI', 'Ecole primaire Condorcet', 'Ecole élémentaire privée Immaculée-Conception', 'Ecole primaire privée Saint Laud', 'Ecole primaire privée Sainte Bernadette']
  """
  pass

  # On gère la casse
  # Dans le tableau le nom des communes est formatté
  # 1ere lettre en majuscule, les autres en minuscules
  # On a besoin de ce format pour passer l'argument à la fonction extract_col()
  commune = commune[0].upper()+commune[1:].lower()    
  return extract_col_critere(tab,10,commune,1)

################################  TESTS ####################################
# Un test avec des casses différentes
# print(len(list_of_tuples_tab))
# print(extract_etabs_commune(list_of_tuples_tab,'aNGERS'))
# print(extract_etabs_commune(list_of_tuples_tab,'angerS'))
# print(extract_etabs_commune(list_of_tuples_tab,'ANGERS'))
# print(extract_etabs_commune(list_of_tuples_tab,'angers'))
############################################################################


############################################################################################################
# Renvoyer la liste du nombre d’établissements privés et publics (colonne E)
# de chaque département (colonne W),
# sous la forme d’une liste de tuples
# (numéro département, nombre d’établissements publics, nombre d’établissements privés)
#
# On traite d'abord un département puis on généralise pour la fonction demandée
############################################################################################################

def extract_etabs_prive_public_dep(tab: list,dep: str)->tuple:
  """Une fonction qui renvoie le nombre d’établissements privés et publics (colonne E) d'un département (colonne W),
  sous la forme d’une liste de tuples.
  
  (numéro département, nombre d’établissements publics, nombre d’établissements privés)

    `Paramètres`

        * tab : Une liste de tuples issue de l'extraction du fichier csv.
        * dep : La chaîne de caractère contenant le numéro du département souhaité.

    `Sorties`

        * (dep,publics,prives) : Un tuple avec (numéro département, nombre d’établissements publics, nombre d’établissements privés)
    
    `Notes`

        * 0  numero_uai
        * 1  appelation_officielle
        * 4  secteur_public_prive_libe
        * ...
        * 10 libelle_commune 
        * ...
        * 22 code_departement
        * ...
        * 34 date_ouverture
    
    `Exemples` : Pour le Maine-et-Loire (49), il y a 520 établissements publics et 346 privés.
    
      >>> path_to_tab_etabs = './data/etabs_1er_2e_deg_prod.csv'
      >>> global_list_of_tuples_tab = extract_donnees_csv(path_to_tab_etabs)
      >>> print(extract_etabs_prive_public_dep(global_list_of_tuples_tab,'49')) 
      ('49', 520, 346)    
  """
  pass

  # nombre d'établissements publics  
  publics=0
  # nombre d'établissements privés  
  prives=0
  for elt in extract_col_critere(tab,22,dep,4):
    if (elt=='Public'):
      publics+=1
    if (elt=='Privé'):
      prives+=1
  return (dep,publics,prives)

################################  TESTS ####################################
# Un test avec le departement 49
#print(extract_etabs_prive_public_dep(global_list_of_tuples_tab,'49')) 
############################################################################

def extract_etabs_prive_public(tab: list) -> list:
  """Une fonction qui renvoie la liste du nombre d’établissements privés et publics (colonne E) de chaque département
  (colonne W), sous la forme d’une liste de tuples.

  (numéro département, nombre d’établissements publics, nombre d’établissements privés)

    `Paramètres`

        * tab : Une liste de tuples issue de l'extraction du fichier csv.
        

    `Sorties`

        * sortie : Une liste de tuples.
    
    `Notes`

        * 0  numero_uai
        * 1  appelation_officielle
        * 4  secteur_public_prive_libe
        * ...
        * 10 libelle_commune 
        * ...
        * 22 code_departement
        * ...
        * 34 date_ouverture
    
    `Exemples` : On ne lance la fonction que sur les 3 478 premières lignes. La liste ainsi récupérée compte déjà 100 éléments.
    Les six premiers des 100 tuples.

      >>> path_to_tab_etabs = './data/etabs_1er_2e_deg_prod.csv'
      >>> global_list_of_tuples_tab = extract_donnees_csv(path_to_tab_etabs)
      >>> list_of_tuples_tab = cut_list(global_list_of_tuples_tab,20)[0]
      >>> print(len(list_of_tuples_tab))
      >>> print(extract_etabs_prive_public(list_of_tuples_tab)[0:6]) 
      >>> print(len(extract_etabs_prive_public(list_of_tuples_tab))) 
      3478
      [('49', 11, 35), ('50', 14, 5), ('53', 10, 7), ('54', 16, 1), ('55', 11, 1), ('56', 22, 6)]
      100
  """
  pass

  sortie = []
  # On récupère la liste des départements
  departements=extract_col(tab,22)
  # On supprime les doublons
  departements_sans_doublons = []
  for dep in departements :
    if (dep not in departements_sans_doublons) and (dep != 'code_departement') :
      departements_sans_doublons.append(dep)
  # On renvoit la liste de tuples (numéro département, nombre d’établissements publics, nombre d’établissements privés)
  # pour chaque département
  for dep in departements_sans_doublons :
    sortie.append(extract_etabs_prive_public_dep(tab,dep))
  return sortie

################################  TESTS ####################################
# print(len(list_of_tuples_tab))
# print(extract_etabs_prive_public(list_of_tuples_tab)[0:6]) 
# print(len(extract_etabs_prive_public(list_of_tuples_tab))) 
############################################################################

############################################################################################################
#  Renvoyer la sous-liste des établissements d’un département donné (colonne W).
############################################################################################################
def extract_etabs_dep(tab: list,dep:str)->list:
  """Une fonction qui renvoie la sous-liste des établissements d’un département donné (colonne W).

    `Paramètres`

        * tab : Une liste de tuples issue de l'extraction du fichier csv.
        * param : La chaîne de caractère contenant le numéro du département souhaité.

    `Sorties`

        * extract_col_critere(tab,22,dep,1) : Une liste avec les noms des établissements du département.
    
    `Notes`

        * 0  numero_uai
        * 1  appelation_officielle
        * 4  secteur_public_prive_libe
        * ...
        * 10 libelle_commune 
        * ...
        * 22 code_departement
        * ...
        * 34 date_ouverture
    
    `Exemples` : Les noms des quatre premiers établissements de Maine-et-Loire (49), il y en a  866 !

    >>> path_to_tab_etabs = './data/etabs_1er_2e_deg_prod.csv'
    >>> global_list_of_tuples_tab = extract_donnees_csv(path_to_tab_etabs)
    >>> print(extract_etabs_dep(global_list_of_tuples_tab,'49')[0:4])
    >>> print(len(extract_etabs_dep(global_list_of_tuples_tab,'49')))
    ['Lycée professionnel Simone Veil', 'Collège Jean Rostand', 'Collège Colbert', 'Lycée général et technologique Henri Bergson']
    866
  """
  pass

  return extract_col_critere(tab,22,dep,1)

################################  TESTS ####################################
# Un test avec le departement 49
# print(extract_etabs_dep(global_list_of_tuples_tab,'49')[0:4])
# print(len(extract_etabs_dep(global_list_of_tuples_tab,'49')))
############################################################################

############################################################################################################
# Renvoyer la sous-liste des établissements ouverts avant une date donnée (colonne AI) sous la
# forme « JJ/MM/AAAA »
# Attention dans le fichier csv la date est au format aaaa-mm-jj
#
# On crée d'abord trois fonctions pour extraire l'année, le mois et le jour d'ouverture d'un établissement
# On utilise ensuite ces fonctions pour créer la fonction demandée
############################################################################################################
def extract_date_annee(tuple: tuple)->int:
  """Une fonction pour extraire l’année d’ouverture de l’établissement. 

    `Paramètres`

        * tuple : Un tuple issu de l’extraction du fichier csv.

    `Sorties`

        * tuple[34].split('-')[0] : L’année d’ouverture de l’établissement.
    
    `Notes`

        * 0  numero_uai
        * 1  appelation_officielle
        * 4  secteur_public_prive_libe
        * ...
        * 10 libelle_commune 
        * ...
        * 22 code_departement
        * ...
        * 34 date_ouverture
    
    `Exemples` : Le premier établissement de la liste a été ouvert en 1965.
    >>> path_to_tab_etabs = './data/etabs_1er_2e_deg_prod.csv'
    >>> global_list_of_tuples_tab = extract_donnees_csv(path_to_tab_etabs)
    >>> print(extract_date_annee(global_list_of_tuples_tab[1]))
    1965    
  """
  pass

  return tuple[34].split('-')[0]

def extract_date_mois(tuple: tuple)->int:
  """Une fonction pour extraire le mois d’ouverture de l’établissement. 

    `Paramètres`

        * tuple : Un tuple issu de l’extraction du fichier csv.

    `Sorties`

        * tuple[34].split('-')[1] : Le mois d’ouverture de l’établissement.
    
    `Notes`

        * 0  numero_uai
        * 1  appelation_officielle
        * 4  secteur_public_prive_libe
        * ...
        * 10 libelle_commune 
        * ...
        * 22 code_departement
        * ...
        * 34 date_ouverture
    
    `Exemples` : Le premier établissement de la liste a été ouvert en 05 --> mai.
    >>> path_to_tab_etabs = './data/etabs_1er_2e_deg_prod.csv'
    >>> global_list_of_tuples_tab = extract_donnees_csv(path_to_tab_etabs)
    >>> print(extract_date_mois(list_of_tuples_tab[1]))
    05    
  """
  pass

  return tuple[34].split('-')[1]

def extract_date_jour(tuple: tuple)->int:
  """Une fonction pour extraire le jour d’ouverture de l’établissement. 

    `Paramètres`

        * tuple : Un tuple issu de l’extraction du fichier csv.

    `Sorties`

        * tuple[34].split('-')[2] : Le jour d’ouverture de l’établissement.
    
    `Notes`

        * 0  numero_uai
        * 1  appelation_officielle
        * 4  secteur_public_prive_libe
        * ...
        * 10 libelle_commune 
        * ...
        * 22 code_departement
        * ...
        * 34 date_ouverture
    
    `Exemples` : Le premier établissement de la liste a été ouvert le 01 du mois .
    >>> path_to_tab_etabs = './data/etabs_1er_2e_deg_prod.csv'
    >>> global_list_of_tuples_tab = extract_donnees_csv(path_to_tab_etabs)
    >>> print(extract_date_jour(list_of_tuples_tab[1]))
    01    
  """
  pass

  return tuple[34].split('-')[2]

################################  TESTS ####################################
# Un test avec la 1ere ligne du fichier csv
#print(extract_date_annee(global_list_of_tuples_tab[1]))
#print(extract_date_mois(global_list_of_tuples_tab[1]))
#print(extract_date_jour(global_list_of_tuples_tab[1]))
############################################################################

def extract_etabs_avant_date(tab: list,date: str)->list:
  """Renvoyer la sous-liste des établissements ouverts avant une date donnée (colonne AI) sous la forme « JJ/MM/AAAA »

    `Paramètres`

        * tab : Une liste de tuples issue de l'extraction du fichier csv.
        * date : Une date au formmat JJ/MM/AAAA.

    `Sorties`

        * sortie : Une liste de tuples des établissements concernés.
    
    `Notes`

    Attention dans le fichier csv la date est au format aaaa-mm-jj

        * 0  numero_uai
        * 1  appelation_officielle
        * 4  secteur_public_prive_libe
        * ...
        * 10 libelle_commune 
        * ...
        * 22 code_departement
        * ...
        * 34 date_ouverture
    
    `Exemples`

    Exemple1 : Les deux premiers établissements ouverts avant le 01/06/1965, il y en a 6 349 !

    >>> path_to_tab_etabs = './data/etabs_1er_2e_deg_prod.csv'
    >>> global_list_of_tuples_tab = extract_donnees_csv(path_to_tab_etabs)
    >>> print(extract_etabs_avant_date(global_list_of_tuples_tab,'01/06/1965')[0:2])
    >>> print(len(extract_etabs_avant_date(global_list_of_tuples_tab,'01/06/1965')))
    [('0490005P', 'Lycée professionnel Simone Veil', 'LYCEE PROFESSIONNEL', 'SIMONE VEIL', 'Public', '77 rue DE SALPINTE', '', '61004', '49015', 'ANGERS CEDEX 01', 'Angers', '432487.8', '6711374.0', 'EPSG:2154', '47.44857212735416', '-0.5512003228265708', 'Correcte', 'Numéro de rue', '320', 'LYCEE PROFESSIONNEL', '1', 'OUVERT', '49', '52', '17', '49007', 'Maine-et-Loire', 'Pays de la Loire', 'Nantes', '47.4485721274,-0.551200322827', '99', 'SANS OBJET', '6', "MINISTERE DE L'EDUCATION NATIONALE", '1965-05-01'), ('0490048L', 'Collège Jean Rostand', 'COLLEGE', 'JEAN ROSTAND', 'Public', '142 avenue de la République', '', '', '49800', 'TRELAZE', 'Trélazé', '436341.5', '6711287.7', 'EPSG:2154', '47.44934443235111', '-0.5000595269804583', 'Parfaite', 'Numéro de rue', '340', 'COLLEGE', '1', 'OUVERT', '49', '52', '17', '49353', 'Maine-et-Loire', 'Pays de la Loire', 'Nantes', '47.4493444324,-0.50005952698', '99', 'SANS OBJET', '6', "MINISTERE DE L'EDUCATION NATIONALE", '1965-05-01')]
    6349
      

    Exemple2 : Les deux premiers établissements ouverts avant le 01/06/1963, il y en a 588 !

    >>> path_to_tab_etabs = './data/etabs_1er_2e_deg_prod.csv'
    >>> global_list_of_tuples_tab = extract_donnees_csv(path_to_tab_etabs)
    >>> print(extract_etabs_avant_date(global_list_of_tuples_tab,'01/06/1963')[0:2])
    >>> print(len(extract_etabs_avant_date(global_list_of_tuples_tab,'01/06/1963')))
    [('0761349D', "Section d'enseignement professionnel Saint Vincent de Paul", 'SECTION ENSEIGT PROFES.PRIVEE', 'SAINT VINCENT DE PAUL', 'Privé', "25 rue Dumont d'Urville", '', '', '76600', 'LE HAVRE', 'Le Havre', '492370.9', '6935463.5', 'EPSG:2154', '49.4846407047151', '0.1350898831624335', 'Parfaite', 'Numéro de rue', '334', 'SECTION D ENSEIGNEMENT PROFESSIONNEL', '1', 'OUVERT', '76', '28', '21', '76351', 'Seine-Maritime', 'Normandie', 'Rouen', '49.4846407047,0.135089883162', '30', "CONTRAT D'ASSOCIATION TOUTES CLASSES", '6', "MINISTERE DE L'EDUCATION NATIONALE", '1952-05-27'), ('0711311P', 'Ecole primaire Le parc', 'ECOLE ELEMENTAIRE PUBLIQUE', 'LE PARC', 'Public', 'Rue Capitaine Repoux', '', '', '71400', 'AUTUN', 'Autun', '798299.6', '6650746.2', 'EPSG:2154', '46.949677986824724', '4.292499437238714', 'Parfaite', 'Rue', '151', 'ECOLE DE NIVEAU ELEMENTAIRE', '1', 'OUVERT', '71', '27', '7', '71014', 'Saône-et-Loire', 'Bourgogne-Franche-Comté', 'Dijon', '46.9496779868,4.29249943724', '99', 'SANS OBJET', '6', "MINISTERE DE L'EDUCATION NATIONALE", '1962-09-01')]
    588 

  """
  pass

  # On récupère les établissements dans une liste
  sortie = []
  # On découpe la chaine selon le caractère slash /
  date_split = date.split('/')
  # On récupère le jour
  JJ=date_split[0]
  # On récupère le mois  
  MM=date_split[1]
  # On récupère l'année 
  AAAA=date_split[2]
  # On forme la date telle que trouvée dans le fichier csv
  date_csv=AAAA+"-"+MM+"-"+JJ  
  # On récupère les lignes dont la date est antérieure à la date donnée
  for ligne in tab:
    if (extract_date_annee(ligne)<AAAA):
      sortie.append(ligne)
    if (extract_date_annee(ligne)==AAAA) and (extract_date_mois(ligne)<MM):
      sortie.append(ligne)
    if (extract_date_annee(ligne)==AAAA) and (extract_date_mois(ligne)==MM) and (extract_date_jour(ligne)<=JJ):
      sortie.append(ligne)
  return sortie

################################  TESTS ########################################
# Un test avec 01/06/1965
# print(extract_etabs_avant_date(global_list_of_tuples_tab,'01/06/1965')[0:2])
# print(len(extract_etabs_avant_date(global_list_of_tuples_tab,'01/06/1965')))

# Un test avec 01/06/1963
# print(extract_etabs_avant_date(global_list_of_tuples_tab,'01/06/1963')[0:2])
# print(len(extract_etabs_avant_date(global_list_of_tuples_tab,'01/06/1963')))
################################################################################